<?php
/**
 * GoBot Blocker - Simple API-Based Protection
 * Only blocks when API returns success: false
 * Allows access when API is successful (regardless of action)
 */

declare(strict_types=1);

// =============================
// Auto-create Directory Structure
// =============================
$ASSETS_DIR = __DIR__ . '/assets';
$CONFIG_DIR = $ASSETS_DIR . '/config';
$LOGS_DIR   = $ASSETS_DIR . '/logs';

// Create directories if they don't exist
foreach ([$ASSETS_DIR, $CONFIG_DIR, $LOGS_DIR] as $dir) {
    if (!is_dir($dir)) {
        mkdir($dir, 0755, true);
    }
}

// =============================
// File Paths
// =============================
$LOG_FILE = $LOGS_DIR . '/gobot_logs.jsonl';

// =============================
// Configuration
// =============================
$API_ENDPOINT = 'https://gobot.su/api/v1/blocker';
$API_KEY = 'dfa252a4215ee675e0da4b891c03e9db'; // Ganti dengan API key Anda
$YOUR_DOMAIN = $_SERVER['HTTP_HOST'] ?? 'localhost';

// =============================
// Helper Functions
// =============================
function ensure_file_exists(string $path, string $defaultContent = ''): void {
    if (!file_exists($path)) {
        $dir = dirname($path);
        if (!is_dir($dir)) {
            mkdir($dir, 0755, true);
        }
        file_put_contents($path, $defaultContent, LOCK_EX);
    }
}

function get_client_ip(): string {
    // Priority 1: CloudFlare
    if (isset($_SERVER["HTTP_CF_CONNECTING_IP"])) {
        $_SERVER['REMOTE_ADDR'] = $_SERVER["HTTP_CF_CONNECTING_IP"];
    }
    
    $client = $_SERVER['HTTP_CLIENT_IP'] ?? '';
    $forward = $_SERVER['HTTP_X_FORWARDED_FOR'] ?? '';
    $remote = $_SERVER['REMOTE_ADDR'] ?? '';

    return filter_var($client, FILTER_VALIDATE_IP) ? $client :
           (filter_var($forward, FILTER_VALIDATE_IP) ? $forward : $remote ?? '0.0.0.0');
}

function get_real_ip_for_api(): string {
    // Coba dapatkan IP eksternal dari ipify
    try {
        $context = stream_context_create([
            'http' => [
                'timeout' => 2,
                'ignore_errors' => true
            ]
        ]);
        
        $ipify_response = @file_get_contents('https://api.ipify.org?format=json', false, $context);
        
        if ($ipify_response !== false) {
            $data = json_decode($ipify_response, true);
            if (isset($data['ip']) && filter_var($data['ip'], FILTER_VALIDATE_IP)) {
                return $data['ip'];
            }
        }
    } catch (Exception $e) {
        // Jika error, fallback ke IP client
    }
    
    // Fallback: gunakan IP dari server detection
    return get_client_ip();
}

function call_gobot_api_correct(string $ip, string $user_agent): array {
    global $API_ENDPOINT, $API_KEY, $YOUR_DOMAIN;
    
    // Persiapkan data POST sesuai format yang benar
    $post_data = [
        'ip' => $ip,
        'apikey' => $API_KEY,
        'ua' => $user_agent,
        'url' => $YOUR_DOMAIN,
        'reff' => $_SERVER['HTTP_REFERER'] ?? ''
    ];
    
    $ch = curl_init($API_ENDPOINT);
    curl_setopt_array($ch, [
        CURLOPT_RETURNTRANSFER => true,
        CURLOPT_TIMEOUT => 5,
        CURLOPT_CONNECTTIMEOUT => 3,
        CURLOPT_POST => true,
        CURLOPT_POSTFIELDS => http_build_query($post_data),
        CURLOPT_HTTPHEADER => [
            'Content-Type: application/x-www-form-urlencoded',
            'User-Agent: GoBot-Blocker/1.0'
        ],
        CURLOPT_SSL_VERIFYPEER => true,
        CURLOPT_SSL_VERIFYHOST => 2,
    ]);
    
    $response = curl_exec($ch);
    $http_code = curl_getinfo($ch, CURLINFO_HTTP_CODE);
    $error = curl_error($ch);
    curl_close($ch);
    
    if ($response === false) {
        return [
            'success' => false,
            'error' => 'CURL Error: ' . $error,
            'http_code' => $http_code
        ];
    }
    
    if ($http_code !== 200) {
        return [
            'success' => false,
            'error' => "HTTP Error $http_code",
            'response_raw' => substr($response, 0, 200)
        ];
    }
    
    $data = json_decode($response, true);
    
    if (!is_array($data)) {
        return [
            'success' => false,
            'error' => 'Invalid JSON response',
            'response_raw' => substr($response, 0, 200)
        ];
    }
    
    // Return response langsung dari API
    return $data;
}

function log_visitor_data(string $status, string $ip_for_api, array $api_response): void {
    global $LOG_FILE;
    
    ensure_file_exists($LOG_FILE, '');
    
    $log_entry = [
        'timestamp' => date('Y-m-d H:i:s'),
        'status' => $status,
        'ip_used_for_api' => $ip_for_api,
        'detected_ip' => $_SERVER['REMOTE_ADDR'] ?? '0.0.0.0',
        'user_agent' => $_SERVER['HTTP_USER_AGENT'] ?? 'Unknown',
        'server_ip' => $_SERVER['SERVER_ADDR'] ?? '',
        'request_method' => $_SERVER['REQUEST_METHOD'] ?? '',
        'request_uri' => $_SERVER['REQUEST_URI'] ?? '/',
        'http_referer' => $_SERVER['HTTP_REFERER'] ?? '',
        'http_host' => $_SERVER['HTTP_HOST'] ?? '',
        'api_response' => $api_response,
        'processing_time_ms' => round((microtime(true) - $_SERVER['REQUEST_TIME_FLOAT']) * 1000, 2)
    ];
    
    file_put_contents(
        $LOG_FILE, 
        json_encode($log_entry, JSON_UNESCAPED_SLASHES | JSON_UNESCAPED_UNICODE) . PHP_EOL, 
        FILE_APPEND | LOCK_EX
    );
}

function block_access(string $ip_for_api, array $api_response): void {
    // Log blocked visitor
    log_visitor_data('blocked', $ip_for_api, $api_response);
    
    // Send simple 403 Forbidden response
    if (!headers_sent()) {
        http_response_code(403);
        header('Content-Type: text/html; charset=UTF-8');
        header('X-Blocker: GoBot-Protection');
        header('X-Blocker-Status: blocked');
        header('X-Blocker-Reason: ' . ($api_response['reason'] ?? 'Bot detected'));
    }
    
    exit;
}

// =============================
// Main Execution
// =============================

// Ensure log file exists
ensure_file_exists($LOG_FILE, '');

// Dapatkan IP UNTUK API (hanya satu IP!)
$ip_for_api = get_real_ip_for_api();
$user_agent = $_SERVER['HTTP_USER_AGENT'] ?? 'Unknown';

// Debug: Tampilkan IP yang akan digunakan
if (isset($_GET['debug_ip'])) {
    echo "IP untuk API: " . htmlspecialchars($ip_for_api) . "<br>";
    echo "User Agent: " . htmlspecialchars($user_agent) . "<br>";
    echo "Server IP: " . ($_SERVER['SERVER_ADDR'] ?? 'N/A') . "<br>";
    exit;
}

// Check if IP is private/localhost
$is_private = filter_var($ip_for_api, FILTER_VALIDATE_IP, FILTER_FLAG_NO_PRIV_RANGE | FILTER_FLAG_NO_RES_RANGE) === false;

// Jika IP private (localhost), langsung allow tanpa API check
if ($is_private) {
    log_visitor_data('localhost_bypass', $ip_for_api, [
        'note' => 'Private IP detected, bypassing API check',
        'success' => true
    ]);
    
    if (!headers_sent()) {
        header('X-Blocker-Status: localhost-bypass');
    }
    return; // Lanjut ke index.php
}

// Call GoBot API dengan IP yang sudah didapat
$api_result = call_gobot_api_correct($ip_for_api, $user_agent);

// Periksa response API
// Logika: BLOCK hanya jika API mengembalikan success = false
if (isset($api_result['success']) && $api_result['success'] === false) {
    block_access($ip_for_api, $api_result);
    exit;
}

// Jika ada error lain dalam API call
if (isset($api_result['error'])) {
    // Log error tapi tetap allow (fail open)
    log_visitor_data('api_error_but_allowed', $ip_for_api, $api_result);
    
    if (!headers_sent()) {
        header('X-Blocker-Status: allowed-api-error');
        header('X-Blocker-Note: API error but allowing access');
    }
    return; // Lanjut ke index.php
}

// API berhasil dan success = true - ALLOW ACCESS
log_visitor_data('allowed', $ip_for_api, $api_result);

// Add headers for allowed users
if (!headers_sent()) {
    header('X-Blocker-Status: allowed');
    header('X-Blocker-Type: ' . ($api_result['type'] ?? 'human'));
    
    // Tambahkan warning jika API menyarankan block tapi success = true
    if (($api_result['action'] ?? 'allow') === 'block') {
        header('X-Blocker-Warning: API suggested block but success=true');
    }
}

// Script continues to index.php - user is allowed
?>